<?php

namespace App\Http\Controllers\Transaction;

use App\Http\Controllers\Controller;
use App\Models\Service;
use App\Models\ServiceDetail;
use App\Models\ServiceStatusHistory;
use App\Models\Product;
use App\Models\Customer;
use App\Models\Branch;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Barryvdh\DomPDF\Facade\Pdf;

class ServiceController extends Controller
{
    public function __construct()
    {
        $this->middleware(['permission:services.index'])->only('index', 'show');
        $this->middleware(['permission:services.create'])->only(['create', 'store']);
        $this->middleware(['permission:services.edit'])->only(['edit', 'update']);
        $this->middleware(['permission:services.delete'])->only('destroy');
    }

    /**
     * Display a listing of services
     */
    public function index()
    {
        $title = "service";
        $services = Service::with(['customer', 'branch', 'technician'])
            ->orderBy('id', 'desc')
            ->get();

        return view('transaction.service.index', compact('title', 'services'));
    }

    /**
     * Show the form for creating a new service
     */
    public function create()
    {
        $title = "service";
        $customers = Customer::orderBy('name', 'asc')->get();
        $branches = Branch::orderBy('name', 'asc')->get();

        // Get technicians - handle jika role technician belum ada
        try {
            $technicians = User::role('technician')->orderBy('name', 'asc')->get();
        } catch (\Exception $e) {
            // Jika role technician belum ada, ambil semua user
            $technicians = User::orderBy('name', 'asc')->get();
        }

        $products = Product::with(['brand', 'category', 'productStocks', 'productPrices'])->get();

        return view('transaction.service.create', compact('title', 'customers', 'branches', 'technicians', 'products'));
    }

    /**
     * Store a newly created service in storage
     */
    public function store(Request $request)
    {
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'branch_id' => 'required|exists:branches,id',
            'service_date' => 'required|date',
            'device_type' => 'required|string',
            'device_brand' => 'required|string',
            'device_model' => 'required|string',
            'complaint' => 'required|string',
        ]);

        DB::beginTransaction();
        try {
            // Generate service number
            $lastService = Service::whereDate('created_at', today())->latest()->first();
            $number = $lastService ? (int) substr($lastService->service_number, -4) + 1 : 1;
            $serviceNumber = 'SRV-' . date('Ymd') . '-' . str_pad($number, 4, '0', STR_PAD_LEFT);

            // Create service
            $service = Service::create([
                'service_number' => $serviceNumber,
                'customer_id' => $request->customer_id,
                'branch_id' => $request->branch_id,
                'technician_id' => $request->technician_id,
                'service_date' => $request->service_date,
                'device_type' => $request->device_type,
                'device_brand' => $request->device_brand,
                'device_model' => $request->device_model,
                'imei' => $request->imei,
                'complaint' => $request->complaint,
                'diagnosis' => $request->diagnosis,
                'estimated_cost' => $request->estimated_cost ?? 0,
                'service_fee' => $request->service_fee ?? 0,
                'down_payment' => $request->down_payment ?? 0,
                'priority' => $request->priority ?? 'normal',
                'estimated_completion' => $request->estimated_completion,
                'notes' => $request->notes,
                'status' => 'menunggu diagnosa',
            ]);

            // Save service details (spare parts) if any
            if ($request->has('products') && is_array($request->products)) {
                $partsCost = 0;
                foreach ($request->products as $item) {
                    if (!empty($item['product_id']) && !empty($item['quantity'])) {
                        $subtotal = $item['quantity'] * $item['unit_price'];
                        $partsCost += $subtotal;

                        ServiceDetail::create([
                            'service_id' => $service->id,
                            'product_id' => $item['product_id'],
                            'quantity' => $item['quantity'],
                            'unit_price' => $item['unit_price'],
                            'subtotal' => $subtotal,
                        ]);
                    }
                }

                // Update parts cost and total
                $service->update([
                    'parts_cost' => $partsCost,
                    'total_cost' => $service->service_fee + $partsCost,
                    'remaining_payment' => ($service->service_fee + $partsCost) - $service->down_payment,
                ]);
            }

            // Create initial status history
            ServiceStatusHistory::create([
                'service_id' => $service->id,
                'status' => 'menunggu diagnosa',
                'notes' => 'Service baru dibuat',
                'user_id' => Auth::id(),
            ]);

            DB::commit();
            return redirect()->route('services.index')->with('success', 'Service berhasil dibuat');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Gagal membuat service: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified service
     */
    public function show($id)
    {
        $title = "service";
        $service = Service::with(['customer', 'branch', 'technician', 'serviceDetails.product', 'statusHistories.user'])
            ->findOrFail($id);

        return view('transaction.service.show', compact('title', 'service'));
    }

    /**
     * Show the form for editing the specified service
     */
    public function edit($id)
    {
        $title = "service";
        $service = Service::with(['serviceDetails'])->findOrFail($id);
        $customers = Customer::orderBy('name', 'asc')->get();
        $branches = Branch::orderBy('name', 'asc')->get();

        // Get technicians - handle jika role technician belum ada
        try {
            $technicians = User::role('technician')->orderBy('name', 'asc')->get();
        } catch (\Exception $e) {
            // Jika role technician belum ada, ambil semua user
            $technicians = User::orderBy('name', 'asc')->get();
        }

        $products = Product::with(['brand', 'category', 'productStocks', 'productPrices'])->get();

        return view('transaction.service.edit', compact('title', 'service', 'customers', 'branches', 'technicians', 'products'));
    }

    /**
     * Update the specified service in storage
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'branch_id' => 'required|exists:branches,id',
            'service_date' => 'required|date',
            'device_type' => 'required|string',
            'device_brand' => 'required|string',
            'device_model' => 'required|string',
            'complaint' => 'required|string',
        ]);

        DB::beginTransaction();
        try {
            $service = Service::findOrFail($id);
            $oldStatus = $service->status;

            // Update service
            $service->update([
                'customer_id' => $request->customer_id,
                'branch_id' => $request->branch_id,
                'technician_id' => $request->technician_id,
                'service_date' => $request->service_date,
                'device_type' => $request->device_type,
                'device_brand' => $request->device_brand,
                'device_model' => $request->device_model,
                'imei' => $request->imei,
                'complaint' => $request->complaint,
                'diagnosis' => $request->diagnosis,
                'estimated_cost' => $request->estimated_cost ?? 0,
                'service_fee' => $request->service_fee ?? 0,
                'down_payment' => $request->down_payment ?? 0,
                'priority' => $request->priority ?? 'normal',
                'estimated_completion' => $request->estimated_completion,
                'notes' => $request->notes,
                'status' => $request->status ?? $service->status,
            ]);

            // Delete old service details
            $service->serviceDetails()->delete();

            // Save new service details (spare parts)
            if ($request->has('products') && is_array($request->products)) {
                $partsCost = 0;
                foreach ($request->products as $item) {
                    if (!empty($item['product_id']) && !empty($item['quantity'])) {
                        $subtotal = $item['quantity'] * $item['unit_price'];
                        $partsCost += $subtotal;

                        ServiceDetail::create([
                            'service_id' => $service->id,
                            'product_id' => $item['product_id'],
                            'quantity' => $item['quantity'],
                            'unit_price' => $item['unit_price'],
                            'subtotal' => $subtotal,
                        ]);
                    }
                }

                // Update parts cost and total
                $service->update([
                    'parts_cost' => $partsCost,
                    'total_cost' => $service->service_fee + $partsCost,
                    'remaining_payment' => ($service->service_fee + $partsCost) - $service->down_payment,
                ]);
            }

            // Create status history if status changed
            if ($oldStatus !== $service->status) {
                ServiceStatusHistory::create([
                    'service_id' => $service->id,
                    'status' => $service->status,
                    'notes' => $request->status_notes ?? 'Status diubah',
                    'user_id' => Auth::id(),
                ]);

                // Set completed_at if status is selesai
                if ($service->status === 'selesai') {
                    $service->update(['completed_at' => now()]);
                }
            }

            DB::commit();
            return redirect()->route('services.index')->with('success', 'Service berhasil diupdate');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Gagal mengupdate service: ' . $e->getMessage());
        }
    }

    /**
     * Update service status
     */
    public function updateStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|string',
            'notes' => 'nullable|string',
        ]);

        DB::beginTransaction();
        try {
            $service = Service::findOrFail($id);
            $oldStatus = $service->status;

            // Update status
            $service->update(['status' => $request->status]);

            // Set completed_at if status is selesai
            if ($request->status === 'selesai' && $oldStatus !== 'selesai') {
                $service->update(['completed_at' => now()]);
            }

            // Create status history
            ServiceStatusHistory::create([
                'service_id' => $service->id,
                'status' => $request->status,
                'notes' => $request->notes ?? 'Status diubah menjadi ' . $request->status,
                'user_id' => Auth::id(),
            ]);

            DB::commit();
            return redirect()->back()->with('success', 'Status service berhasil diupdate');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Gagal mengupdate status: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified service from storage
     */
    public function destroy($id)
    {
        try {
            $service = Service::findOrFail($id);
            $service->delete();

            return redirect()->route('services.index')->with('success', 'Service berhasil dihapus');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Gagal menghapus service: ' . $e->getMessage());
        }
    }

    /**
     * Print service receipt
     */
    public function print($id)
    {
        $service = Service::with(['customer', 'branch', 'technician', 'serviceDetails.product'])
            ->findOrFail($id);

        $pdf = Pdf::loadView('transaction.service.print', compact('service'));
        return $pdf->stream('service-' . $service->service_number . '.pdf');
    }
}
