<?php

namespace App\Http\Controllers\Customer;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Customer;
use App\Models\Sale;
use App\Models\Service;
use Illuminate\Support\Facades\Auth;

class CustomerDashboardController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'role:customer']);
    }

    public function index()
    {
        $user = Auth::user();

        // Cari customer berdasarkan user_id atau email
        $customer = Customer::where('user_id', $user->id)
                           ->orWhere('email', $user->email)
                           ->first();

        if (!$customer) {
            return redirect()->route('login')->with('error', 'Data customer tidak ditemukan.');
        }

        // Ambil data pembelian (sales)
        $purchases = Sale::with(['saleDetails.product', 'branch'])
                        ->where('customer_id', $customer->id)
                        ->orderBy('sale_date', 'desc')
                        ->get();

        // Ambil data service
        $services = Service::with(['branch', 'technician', 'serviceDetails.product', 'statusHistories'])
                          ->where('customer_id', $customer->id)
                          ->orderBy('service_date', 'desc')
                          ->get();

        // Service yang sedang aktif (belum selesai)
        $activeServices = $services->whereNotIn('status', ['selesai', 'diambil', 'dibatalkan']);

        // Statistik
        $stats = [
            'total_purchases' => $purchases->count(),
            'total_spent' => $purchases->sum('total_amount'),
            'total_services' => $services->count(),
            'active_services' => $activeServices->count(),
        ];

        return view('customer.dashboard', compact('customer', 'purchases', 'services', 'activeServices', 'stats'));
    }

    public function serviceDetail($id)
    {
        $user = Auth::user();

        // Cari customer berdasarkan user_id atau email
        $customer = Customer::where('user_id', $user->id)
                           ->orWhere('email', $user->email)
                           ->first();

        if (!$customer) {
            return redirect()->route('customer.dashboard')->with('error', 'Data customer tidak ditemukan.');
        }

        // Ambil service detail dengan validasi bahwa service ini milik customer yang login
        $service = Service::with(['branch', 'technician', 'serviceDetails.product', 'statusHistories.user'])
                         ->where('customer_id', $customer->id)
                         ->findOrFail($id);

        return view('customer.service-detail', compact('service', 'customer'));
    }
}
