# 🔧 Fix: Auto-Load Harga Beli di Form Pembelian

## ✅ Status: SELESAI

---

## 🐛 Masalah Sebelumnya

**User Report:**
> "HARGA BELI DI MENU PEMBELIAN MASIH 0. PADAHAL HARGA BELI DI MENU PRODUCT ITU 1000000"

### Detail Masalah:
- User sudah mengatur Harga Beli produk di menu Product (Rp 1.000.000)
- Ketika membuat pembelian dan memilih produk, field Harga Beli menampilkan 0
- User harus mengisi harga secara manual setiap kali

### Root Cause:
- Form pembelian tidak memiliki fungsi untuk mengambil harga dari database
- JavaScript hanya mengambil nama dan satuan produk
- Tidak ada AJAX call ke server untuk fetch data harga

---

## ✨ Solusi yang Diterapkan

### 1. **API Endpoint Baru untuk Get Product Price**

**File:** `app/Http/Controllers/Transaction/PurchaseController.php`

**Method baru ditambahkan:**
```php
/**
 * Get product price for specific branch
 */
public function getProductPrice(Request $request)
{
    $productId = $request->input('product_id');
    $branchId = $request->input('branch_id');

    if (!$productId || !$branchId) {
        return response()->json([
            'success' => false,
            'message' => 'Product ID dan Branch ID harus diisi'
        ], 400);
    }

    // Get the latest price for the product at the specific branch
    $productPrice = \App\Models\ProductPrice::where('product_id', $productId)
        ->where('branch_id', $branchId)
        ->orderBy('effective_date', 'desc')
        ->first();

    if ($productPrice) {
        return response()->json([
            'success' => true,
            'purchase_price' => $productPrice->purchase_price,
            'selling_price' => $productPrice->selling_price,
        ]);
    }

    return response()->json([
        'success' => false,
        'message' => 'Harga belum diatur untuk produk ini di cabang yang dipilih'
    ], 404);
}
```

**Cara Kerja:**
- Menerima `product_id` dan `branch_id` dari request
- Query ke tabel `product_prices` untuk mendapatkan harga terbaru (berdasarkan `effective_date`)
- Return JSON dengan `purchase_price` dan `selling_price`
- Jika tidak ada data, return error message

---

### 2. **Route Baru**

**File:** `routes/web.php`

**Route ditambahkan:**
```php
Route::get('/purchase/get-product-price', [App\Http\Controllers\Transaction\PurchaseController::class, 'getProductPrice'])
    ->name('purchase.get-product-price');
```

**URL:** `/admin/purchase/get-product-price?product_id=1&branch_id=2`

---

### 3. **Update JavaScript di Form Pembelian**

**File:** `resources/views/transaction/purchase/create.blade.php`

#### A. **Validasi Cabang Harus Dipilih Dulu**
```javascript
// Get selected branch
const branchId = $('select[name="branch_id"]').val();
if (!branchId) {
    alert('Silakan pilih cabang terlebih dahulu!');
    $(this).val('').trigger('change');
    return;
}
```

#### B. **AJAX Call untuk Fetch Harga**
```javascript
// Fetch product price from server
$.ajax({
    url: '{{ route("purchase.get-product-price") }}',
    method: 'GET',
    data: {
        product_id: productId,
        branch_id: branchId
    },
    success: function(response) {
        if (response.success) {
            addProductRow(productId, productName, productUnit, response.purchase_price);
        } else {
            // If no price found, add with 0 and show warning
            alert(response.message + '\nSilakan isi harga secara manual.');
            addProductRow(productId, productName, productUnit, 0);
        }
        addedProducts.add(productId);
    },
    error: function(xhr) {
        alert('Gagal mengambil harga produk. Silakan isi harga secara manual.');
        addProductRow(productId, productName, productUnit, 0);
        addedProducts.add(productId);
    }
});
```

#### C. **Update Function `addProductRow`**
```javascript
// BEFORE
function addProductRow(productId, productName, productUnit) {
    ...
    value="0" // Static 0
    ...
}

// AFTER
function addProductRow(productId, productName, productUnit, purchasePrice = 0) {
    ...
    value="${purchasePrice}" // Dynamic from server
    ...
}
```

---

## 🎯 Cara Kerja Lengkap

### Flow Diagram:
```
1. User memilih CABANG
2. User memilih PRODUK dari dropdown
3. JavaScript mengambil product_id dan branch_id
4. AJAX call ke server: GET /purchase/get-product-price
5. Server query product_prices table
6. Server return purchase_price
7. JavaScript populate field Harga Beli dengan nilai dari server
8. User bisa langsung ubah qty atau simpan
```

### Contoh Request/Response:

**Request:**
```
GET /admin/purchase/get-product-price?product_id=1&branch_id=2
```

**Response (Success):**
```json
{
    "success": true,
    "purchase_price": 1000000,
    "selling_price": 1200000
}
```

**Response (Not Found):**
```json
{
    "success": false,
    "message": "Harga belum diatur untuk produk ini di cabang yang dipilih"
}
```

---

## 📝 Testing Scenario

### Test Case 1: Produk dengan Harga Sudah Diatur
```
GIVEN: Product ID 1 memiliki harga beli Rp 1.000.000 di cabang Jakarta
WHEN: User pilih cabang Jakarta → Pilih Product ID 1
THEN: Field Harga Beli otomatis terisi Rp 1.000.000
```

### Test Case 2: Produk Tanpa Harga di Cabang
```
GIVEN: Product ID 2 tidak memiliki harga di cabang Bandung
WHEN: User pilih cabang Bandung → Pilih Product ID 2
THEN:
  - Alert muncul: "Harga belum diatur untuk produk ini di cabang yang dipilih. Silakan isi harga secara manual."
  - Field Harga Beli tetap 0
  - User harus isi manual
```

### Test Case 3: Belum Pilih Cabang
```
GIVEN: User belum memilih cabang
WHEN: User langsung pilih produk
THEN:
  - Alert muncul: "Silakan pilih cabang terlebih dahulu!"
  - Produk tidak ditambahkan ke tabel
```

---

## 🚀 Keunggulan Solusi Ini

### 1. **Branch-Specific Pricing**
- Harga berbeda per cabang bisa dihandle
- Otomatis ambil harga sesuai cabang yang dipilih

### 2. **Latest Price**
- Selalu ambil harga terbaru berdasarkan `effective_date`
- Support multi-period pricing

### 3. **Fallback Mechanism**
- Jika tidak ada harga, user tetap bisa input manual
- Tidak blocking workflow

### 4. **User-Friendly**
- Alert yang informatif
- Validasi step-by-step (pilih cabang dulu)

### 5. **No Page Reload**
- AJAX-based, seamless experience
- Fast response

---

## 🔄 Perbandingan Before & After

### BEFORE:
```
1. User pilih produk
2. Harga muncul 0
3. User harus cek di menu Product
4. User copy harga manual
5. User paste ke form pembelian
```

### AFTER:
```
1. User pilih cabang
2. User pilih produk
3. Harga otomatis terisi ✅
4. User bisa langsung save
```

**Time Saved:** ~30 detik per produk × jumlah item pembelian

---

## ⚙️ Database Schema Reference

### Table: `product_prices`
```sql
CREATE TABLE product_prices (
    id BIGINT UNSIGNED PRIMARY KEY,
    product_id BIGINT UNSIGNED,
    branch_id BIGINT UNSIGNED,
    purchase_price DECIMAL(15,2),
    selling_price DECIMAL(15,2),
    effective_date DATE,
    created_at TIMESTAMP,
    updated_at TIMESTAMP
);
```

**Index untuk Performance:**
- `(product_id, branch_id, effective_date DESC)` - untuk query cepat harga terbaru

---

## 🎨 UI/UX Improvements

### Alert Messages:
1. **"Silakan pilih cabang terlebih dahulu!"**
   - Muncul jika user langsung pilih produk tanpa pilih cabang
   - Prevents error dan guide user

2. **"Harga belum diatur untuk produk ini di cabang yang dipilih\nSilakan isi harga secara manual."**
   - Muncul jika tidak ada data harga
   - User tahu harus manual input

3. **"Gagal mengambil harga produk. Silakan isi harga secara manual."**
   - Muncul jika terjadi error network/server
   - Fallback to manual input

---

## 📊 Data Flow

```
┌─────────────┐
│  User Input │
│  (Browser)  │
└──────┬──────┘
       │ 1. Select product + branch
       ▼
┌─────────────────┐
│ JavaScript AJAX │
│   (Frontend)    │
└──────┬──────────┘
       │ 2. GET /purchase/get-product-price
       ▼
┌──────────────────┐
│ PurchaseController│
│   (Backend)       │
└──────┬───────────┘
       │ 3. Query DB
       ▼
┌──────────────────┐
│  product_prices  │
│     (MySQL)      │
└──────┬───────────┘
       │ 4. Return price
       ▼
┌──────────────────┐
│ JSON Response    │
│ {purchase_price} │
└──────┬───────────┘
       │ 5. Populate input field
       ▼
┌──────────────────┐
│  Form Updated    │
│  (User sees Rp)  │
└──────────────────┘
```

---

## 🔍 Troubleshooting

### Issue 1: Harga Masih 0
**Kemungkinan:**
- Data harga belum diinput di menu Product
- Harga diinput tapi untuk cabang yang berbeda
- Field `effective_date` kosong atau di masa depan

**Solusi:**
1. Buka menu **Product > Product**
2. Klik **Edit** pada produk yang dimaksud
3. Tab **Harga Per Cabang**
4. Pastikan ada data harga untuk cabang yang dipilih
5. Pastikan `effective_date` ≤ hari ini

### Issue 2: Alert "Gagal mengambil harga"
**Kemungkinan:**
- Server error
- Route tidak terdaftar
- Permission issue

**Solusi:**
1. Cek Laravel log: `storage/logs/laravel.log`
2. Cek route terdaftar: `php artisan route:list | grep purchase`
3. Cek permission: `purchases.create` harus ada

### Issue 3: Alert "Pilih cabang dulu"
**Ini bukan error!** Ini fitur validasi.

**Workflow yang benar:**
1. Pilih Supplier
2. **Pilih Cabang** ← PENTING!
3. Pilih Produk

---

## 📚 Files Modified

### 1. Controller
- **File:** `app/Http/Controllers/Transaction/PurchaseController.php`
- **Changes:** Added `getProductPrice()` method

### 2. Routes
- **File:** `routes/web.php`
- **Changes:** Added route for `purchase.get-product-price`

### 3. View
- **File:** `resources/views/transaction/purchase/create.blade.php`
- **Changes:**
  - Added branch validation
  - Added AJAX call
  - Updated `addProductRow()` function to accept price parameter

---

## ✅ Checklist Completion

- [x] Create API endpoint `getProductPrice()`
- [x] Add route for the endpoint
- [x] Add branch validation in JavaScript
- [x] Implement AJAX call to fetch price
- [x] Update `addProductRow()` to accept price
- [x] Handle success response
- [x] Handle error/not found response
- [x] Add user-friendly alerts
- [x] Test with existing data
- [x] Document the feature

---

## 🎉 Kesimpulan

**Masalah:** Harga beli tidak auto-load dari master data product

**Solusi:** Implementasi AJAX-based price fetching

**Hasil:**
- ✅ Harga otomatis terisi saat pilih produk
- ✅ Support multi-branch pricing
- ✅ Fallback ke manual input jika diperlukan
- ✅ User experience lebih baik
- ✅ Time saving untuk data entry

**Status:** SELESAI & SIAP DIGUNAKAN ✅

---

**Created:** 15 Desember 2025
**Version:** 1.0
**Bug Fixed:** Purchase Price Auto-Load Issue
**Framework:** Laravel 12 + jQuery + AJAX
