<?php

namespace App\Http\Controllers\Master;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Customer;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class CustomerController extends Controller
{
    public function __construct()
    {
        $this->middleware(['permission:customers.index'])->only('index');
        $this->middleware(['permission:customers.create'])->only('save');
        $this->middleware(['permission:customers.edit'])->only('update');
        $this->middleware(['permission:customers.delete'])->only('delete');
    }

    public function index()
    {
        $title = "customer";
        $customers = Customer::orderBy('id', 'desc')->get();

        return view('master.customer', compact('title','customers'));
    }

    public function save(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|unique:customers',
            'email' => 'required|email|unique:users,email'
        ]);

        // Buat customer
        $customer = Customer::create([
            'name' => $request->input('name'),
            'phone' => $request->input('phone'),
            'address' => $request->input('address'),
            'email' => $request->input('email'),
        ]);

        if($customer){
            // Buat user akun otomatis untuk customer
            $defaultPassword = 'customer123'; // Password default

            $user = User::create([
                'name' => $customer->name,
                'email' => $customer->email,
                'password' => Hash::make($defaultPassword),
            ]);

            // Assign role customer ke user (jika ada role customer)
            if($user->hasRole('customer') == false) {
                $user->assignRole('customer');
            }

            // Update customer dengan user_id
            $customer->update(['user_id' => $user->id]);

            //redirect dengan pesan sukses
            return redirect()->route('customer')->with(['success' => 'Data Customer dan User Akun Berhasil Disimpan! Password default: ' . $defaultPassword]);
        }else{
            //redirect dengan pesan error
            return redirect()->route('customer')->with(['error' => 'Data Gagal Disimpan!']);
        }
    }

    public function update(Request $request, Customer $customer)
    {
        $this->validate($request, [
            'name' => 'required|unique:customers,name,'.$customer->id
        ]);

        $customer = Customer::findOrFail($customer->id);
        $customer->update([
            'name' => $request->input('name'),
            'phone' => $request->input('phone'),
            'address' => $request->input('address'),
            'email' => $request->input('email')
        ]);

        if($customer){
            //redirect dengan pesan sukses
            return redirect()->route('customer')->with(['success' => 'Data Berhasil Disimpan!']);
        }else{
            //redirect dengan pesan error
            return redirect()->route('customer')->with(['error' => 'Data Gagal Disimpan!']);
        }
    }

    public function delete(Customer $customer)
    {
        $customer->delete();

        return redirect()->route('customer')->with('success', 'Data Berhasil Dihapus');
    }

    /**
     * Download template Excel untuk import customer
     */
    public function downloadTemplate()
    {
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Set header
        $sheet->setCellValue('A1', 'Nama Customer');
        $sheet->setCellValue('B1', 'No. Telepon');
        $sheet->setCellValue('C1', 'Email');
        $sheet->setCellValue('D1', 'Alamat');

        // Style header
        $sheet->getStyle('A1:D1')->getFont()->setBold(true);
        $sheet->getStyle('A1:D1')->getFill()
            ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
            ->getStartColor()->setARGB('FFCCCCCC');

        // Set column width
        $sheet->getColumnDimension('A')->setWidth(30);
        $sheet->getColumnDimension('B')->setWidth(20);
        $sheet->getColumnDimension('C')->setWidth(30);
        $sheet->getColumnDimension('D')->setWidth(50);

        // Add example data
        $sheet->setCellValue('A2', 'Contoh Customer 1');
        $sheet->setCellValue('B2', '081234567890');
        $sheet->setCellValue('C2', 'customer1@example.com');
        $sheet->setCellValue('D2', 'Jl. Contoh No. 123, Jakarta');
        $sheet->setCellValue('A3', 'Contoh Customer 2');
        $sheet->setCellValue('B3', '081234567891');
        $sheet->setCellValue('C3', 'customer2@example.com');
        $sheet->setCellValue('D3', 'Jl. Contoh No. 456, Bandung');

        // Create Excel file
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        $fileName = 'template_import_customer.xlsx';

        // Set headers for download
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $fileName . '"');
        header('Cache-Control: max-age=0');

        $writer->save('php://output');
        exit;
    }

    /**
     * Import customer dari Excel dengan auto-create user account
     */
    public function importExcel(Request $request)
    {
        $this->validate($request, [
            'excel_file' => 'required|mimes:xlsx,xls|max:2048',
        ]);

        try {
            $file = $request->file('excel_file');
            $spreadsheet = \PhpOffice\PhpSpreadsheet\IOFactory::load($file->getPathname());
            $sheet = $spreadsheet->getActiveSheet();
            $rows = $sheet->toArray();

            // Cek apakah file kosong
            if (count($rows) < 2) {
                return redirect()->back()->with('error', 'File Excel kosong atau format tidak sesuai!');
            }

            $successCount = 0;
            $errorCount = 0;
            $errors = [];
            $defaultPassword = 'customer123'; // Password default untuk semua customer

            // Mulai dari baris 2 (index 1) - setelah header
            for ($i = 1; $i < count($rows); $i++) {
                $row = $rows[$i];

                // Skip jika baris kosong
                if (empty($row[0]) && empty($row[1]) && empty($row[2]) && empty($row[3])) {
                    continue;
                }

                $rowNumber = $i + 1;
                $customerName = isset($row[0]) ? trim($row[0]) : '';
                $phone = isset($row[1]) ? trim($row[1]) : '';
                $email = isset($row[2]) ? trim($row[2]) : '';
                $address = isset($row[3]) ? trim($row[3]) : '';

                // Validasi data wajib
                if (empty($customerName)) {
                    $errors[] = "Baris {$rowNumber}: Nama customer tidak boleh kosong";
                    $errorCount++;
                    continue;
                }

                if (empty($email)) {
                    $errors[] = "Baris {$rowNumber}: Email tidak boleh kosong";
                    $errorCount++;
                    continue;
                }

                // Validasi format email
                if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                    $errors[] = "Baris {$rowNumber}: Format email tidak valid";
                    $errorCount++;
                    continue;
                }

                // Cek apakah customer sudah ada berdasarkan email
                $existingCustomer = Customer::where('email', $email)->first();

                if ($existingCustomer) {
                    // Update customer yang sudah ada
                    $existingCustomer->update([
                        'name' => $customerName,
                        'phone' => $phone,
                        'address' => $address
                    ]);
                    $successCount++;
                } else {
                    // Cek apakah email sudah digunakan di tabel users
                    $existingUser = User::where('email', $email)->first();

                    if ($existingUser) {
                        $errors[] = "Baris {$rowNumber}: Email '{$email}' sudah terdaftar sebagai user";
                        $errorCount++;
                        continue;
                    }

                    // Buat customer baru
                    $customer = Customer::create([
                        'name' => $customerName,
                        'phone' => $phone,
                        'email' => $email,
                        'address' => $address
                    ]);

                    // Buat user account otomatis untuk customer
                    $user = User::create([
                        'name' => $customer->name,
                        'email' => $customer->email,
                        'password' => Hash::make($defaultPassword),
                    ]);

                    // Assign role customer ke user
                    if($user->hasRole('customer') == false) {
                        $user->assignRole('customer');
                    }

                    // Update customer dengan user_id
                    $customer->update(['user_id' => $user->id]);

                    $successCount++;
                }
            }

            $message = "Import selesai! Berhasil: {$successCount}, Gagal: {$errorCount}";
            $message .= "\n\nPassword default untuk semua customer: {$defaultPassword}";

            if ($errorCount > 0) {
                $message .= "\n\nDetail error:\n" . implode("\n", array_slice($errors, 0, 10));
                if (count($errors) > 10) {
                    $message .= "\n... dan " . (count($errors) - 10) . " error lainnya";
                }
            }

            if ($successCount > 0) {
                return redirect()->route('customer')->with('success', $message);
            } else {
                return redirect()->route('customer')->with('error', $message);
            }

        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Gagal import Excel: ' . $e->getMessage());
        }
    }

}

