<?php

namespace App\Http\Controllers\Master;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Category;

class CategoriesController extends Controller
{
    public function __construct()
    {
        $this->middleware(['permission:branches.index'])->only('index');
        $this->middleware(['permission:branches.create'])->only('save');
        $this->middleware(['permission:branches.edit'])->only('update');
        $this->middleware(['permission:branches.delete'])->only('delete');
    }

    public function index()
    {
        $title = "category";
        $categories = Category::orderBy('id', 'desc')->get();

        return view('master.category', compact('title','categories'));
    }

    public function save(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|unique:categories'
        ]);

        $category = Category::create([
            'name' => $request->input('name'),
            'description' => $request->input('description')
        ]);

        if($category){
            //redirect dengan pesan sukses
            return redirect()->route('category')->with(['success' => 'Data Berhasil Disimpan!']);
        }else{
            //redirect dengan pesan error
            return redirect()->route('category')->with(['error' => 'Data Gagal Disimpan!']);
        }
    }

    public function update(Request $request, Category $category)
    {
        $this->validate($request, [
            'name' => 'required|unique:categories,name,'.$category->id
        ]);

        $category = Category::findOrFail($category->id);
        $category->update([
            'name' => $request->input('name'),
            'description' => $request->input('description')
        ]);

        if($category){
            //redirect dengan pesan sukses
            return redirect()->route('category')->with(['success' => 'Data Berhasil Disimpan!']);
        }else{
            //redirect dengan pesan error
            return redirect()->route('category')->with(['error' => 'Data Gagal Disimpan!']);
        }
    }

    public function delete(Category $category)
    {
        $category->delete();

        return redirect()->route('category')->with('success', 'Data Berhasil Dihapus');
    }

    /**
     * Download template Excel untuk import kategori
     */
    public function downloadTemplate()
    {
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Set header
        $sheet->setCellValue('A1', 'Nama Kategori');
        $sheet->setCellValue('B1', 'Deskripsi');

        // Style header
        $sheet->getStyle('A1:B1')->getFont()->setBold(true);
        $sheet->getStyle('A1:B1')->getFill()
            ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
            ->getStartColor()->setARGB('FFCCCCCC');

        // Set column width
        $sheet->getColumnDimension('A')->setWidth(30);
        $sheet->getColumnDimension('B')->setWidth(50);

        // Add example data
        $sheet->setCellValue('A2', 'Contoh Kategori 1');
        $sheet->setCellValue('B2', 'Deskripsi contoh kategori 1');
        $sheet->setCellValue('A3', 'Contoh Kategori 2');
        $sheet->setCellValue('B3', 'Deskripsi contoh kategori 2');

        // Create Excel file
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        $fileName = 'template_import_kategori.xlsx';

        // Set headers for download
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $fileName . '"');
        header('Cache-Control: max-age=0');

        $writer->save('php://output');
        exit;
    }

    /**
     * Import kategori dari Excel
     */
    public function importExcel(Request $request)
    {
        $this->validate($request, [
            'excel_file' => 'required|mimes:xlsx,xls|max:2048',
        ]);

        try {
            $file = $request->file('excel_file');
            $spreadsheet = \PhpOffice\PhpSpreadsheet\IOFactory::load($file->getPathname());
            $sheet = $spreadsheet->getActiveSheet();
            $rows = $sheet->toArray();

            // Cek apakah file kosong
            if (count($rows) < 2) {
                return redirect()->back()->with('error', 'File Excel kosong atau format tidak sesuai!');
            }

            $successCount = 0;
            $errorCount = 0;
            $errors = [];

            // Mulai dari baris 2 (index 1) - setelah header
            for ($i = 1; $i < count($rows); $i++) {
                $row = $rows[$i];

                // Skip jika baris kosong
                if (empty($row[0]) && empty($row[1])) {
                    continue;
                }

                $rowNumber = $i + 1;
                $categoryName = isset($row[0]) ? trim($row[0]) : '';
                $description = isset($row[1]) ? trim($row[1]) : '';

                // Validasi data wajib
                if (empty($categoryName)) {
                    $errors[] = "Baris {$rowNumber}: Nama kategori tidak boleh kosong";
                    $errorCount++;
                    continue;
                }

                // Cek apakah kategori sudah ada
                $existingCategory = Category::where('name', $categoryName)->first();

                if ($existingCategory) {
                    // Update kategori yang sudah ada
                    $existingCategory->update([
                        'description' => $description
                    ]);
                    $successCount++;
                } else {
                    // Buat kategori baru
                    Category::create([
                        'name' => $categoryName,
                        'description' => $description
                    ]);
                    $successCount++;
                }
            }

            $message = "Import selesai! Berhasil: {$successCount}, Gagal: {$errorCount}";

            if ($errorCount > 0) {
                $message .= "\n\nDetail error:\n" . implode("\n", array_slice($errors, 0, 10));
                if (count($errors) > 10) {
                    $message .= "\n... dan " . (count($errors) - 10) . " error lainnya";
                }
            }

            if ($successCount > 0) {
                return redirect()->route('category')->with('success', $message);
            } else {
                return redirect()->route('category')->with('error', $message);
            }

        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Gagal import Excel: ' . $e->getMessage());
        }
    }

}
